#!/usr/bin/env bash
set -euo pipefail

# --- Paths ---
ROOT="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
CFG="$ROOT/nprog/auto_max2.cfg"
SVF_240="${ROOT}/rtl/EPM240_bitstream.svf"
SVF_570="${ROOT}/rtl/EPM570_bitstream.svf"   # optional; only if you ship one

OPENOCD_BIN="${OPENOCD_BIN:-openocd}"
OPENOCD_SCRIPTS="${OPENOCD_SCRIPTS:-/usr/share/openocd/scripts}"
if ! command -v "$OPENOCD_BIN" >/dev/null 2>&1; then
  echo "ERROR: openocd not found (set OPENOCD_BIN=...)" >&2; exit 1
fi

# --- Detect Raspberry Pi peripheral base ---
# Pi 0/1  : 0x20000000
# Pi 2/3/Zero 2 W: 0x3F000000
# Pi 4/5  : 0xFE000000
PERIPH_BASE="0x3F000000"
if grep -aq "BCM2708" /proc/cpuinfo 2>/dev/null; then
  PERIPH_BASE="0x20000000"
elif grep -aqE "BCM2711|BCM2712" /proc/cpuinfo 2>/dev/null; then
  PERIPH_BASE="0xFE000000"
fi

# Allow manual override: PERIPH_BASE=0xFE000000 ./mprog_auto.sh
PERIPH_BASE="${PERIPH_BASE_OVERRIDE:-${PERIPH_BASE}}"

# --- Probe the chain to learn the IDCODE ---
SCAN_OUT="$(sudo env OPENOCD_SCRIPTS="$OPENOCD_SCRIPTS" \
  "$OPENOCD_BIN" -f "$CFG" \
  -c "set PERIPH_BASE $PERIPH_BASE; init; scan_chain; shutdown" 2>&1 || true)"

echo "$SCAN_OUT"

if grep -q "0x020a10dd" <<<"$SCAN_OUT"; then
  CHIP="EPM240"
  SVF="$SVF_240"
elif grep -q "0x020a20dd" <<<"$SCAN_OUT"; then
  CHIP="EPM570"
  SVF="$SVF_570"
else
  echo "ERROR: Unknown/unsupported MAX II IDCODE (wanted EPM240/EPM570)." >&2
  exit 2
fi

if [[ ! -f "$SVF" ]]; then
  echo "ERROR: SVF file for $CHIP not found: $SVF" >&2
  exit 3
fi

echo "Detected $CHIP. Using SVF: $SVF"
# Default to 100 kHz; allow override: SPEED_KHZ=50 ./mprog_auto.sh
SPEED_KHZ="${SPEED_KHZ:-100}"

# --- Program with visible progress ---
VERBOSITY="${VERBOSITY:-quiet}"   # or 'progress'
sudo env OPENOCD_SCRIPTS="$OPENOCD_SCRIPTS" \
  "$OPENOCD_BIN" -f "$CFG" \
  -c "set PERIPH_BASE $PERIPH_BASE; adapter speed $SPEED_KHZ; init; svf \"$SVF\" $VERBOSITY; shutdown"
